/*

ServiceWindowManager.cs, Version 1.0, June 2004

Copyright 2004 Brian Loesgen, Neudesic LLC.

You may use this code royalty-free, re-distribute it, modify it or deploy it at client sites, 
provided that this header information is included and remains intact.

*/

using System;
using System.Collections;
using System.Diagnostics;
using System.Xml;

using Microsoft.BizTalk.ExplorerOM;

namespace Neudesic.BTSUtilities
{
	/// <summary>
	/// This class manages the ReceiveLocation objects
	/// </summary>
	public class ServiceWindowManager
	{
		ArrayList arrLocations = new ArrayList();
		BtsCatalogExplorer BTSConfig = new BtsCatalogExplorer();

		/// <summary>
		/// flag: write trace messages to the application event log?
		/// </summary>
		public bool LoggingEnabled
		{
			get {return ReceiveLocation.LoggingEnabled;}
			set {ReceiveLocation.LoggingEnabled = value;}
		}


		/// <summary>
		/// Constructor. Read in all the configuration data and set up the ReceiveLocation objects
		/// </summary>
		/// <param name="config">Connnection string to reach the BizTalk Management database. Required.</param>
		public ServiceWindowManager(string config)
		{
			LoggingEnabled = true;

			XmlDocument doc = new XmlDocument();

			try
			{
				// load config file
				doc.Load(config);
			}
			catch (Exception ex)
			{
				throw new Exception("Unable to load config file. Error: " + ex.Message);
			}

			EventLog.WriteEntry("ServiceWindowManager loading config file " + config,"");
			
			// connect to the management db
			try
			{
				BTSConfig.ConnectionString = (string) doc.GetElementsByTagName("ServiceWindows")[0].Attributes["connection"].Value;
			}
			catch (Exception ex)
			{
				throw new Exception("Unable to connect to BizTalk management database. Error: " + ex.Message);			
			}

			// get all the data
			XmlNodeList locations = doc.GetElementsByTagName("ReceiveLocation");

			foreach (XmlNode node in locations)
			{
				bool create = true;

				if (node.Attributes["active"]!=null && node.Attributes["active"].Value.ToUpper()=="FALSE")
					create = false;
				
				if (create)
				{

					ReceiveLocation rl = new ReceiveLocation(node.Attributes["port"].Value, node.Attributes["name"].Value, Convert.ToDouble(node.Attributes["interval"].Value), Convert.ToDouble(node.Attributes["duration"].Value));

					rl.BTSConfig = BTSConfig;

					bool anyErrors = false;

					if (node.Attributes["startAt"]!=null)
						rl.SetStartTime(node.Attributes["startAt"].Value);

					if (node.Attributes["enableConditionAssembly"]!=null  && node.Attributes["enableConditionClass"]!=null && node.Attributes["enableConditionMethod"]!=null)
					{
						if (!rl.SetCondition(node.Attributes["enableConditionAssembly"].Value, node.Attributes["enableConditionClass"].Value, node.Attributes["enableConditionMethod"].Value, true))
						{
							// Could not create the enableCondition object, this is a fatal error, report it in the event log
							System.Diagnostics.EventLog.WriteEntry("BTSServiceWindows: Receive Location NOT created!","Unable to create object for EnableLocation of "+rl.Port+":"+rl.Name + ". Is the condition assembly specified in the configuration file in the GAC?",System.Diagnostics.EventLogEntryType.Error);
							anyErrors = true;
						}
					}
					if (node.Attributes["disableConditionAssembly"]!=null  && node.Attributes["disableConditionClass"]!=null && node.Attributes["disableConditionMethod"]!=null)
					{
						if (!rl.SetCondition(node.Attributes["disableConditionAssembly"].Value, node.Attributes["disableConditionClass"].Value, node.Attributes["disableConditionMethod"].Value, false))
						{
							// Could not create the disableCondition object, this is a fatal error, report it in the event log
							System.Diagnostics.EventLog.WriteEntry("BTSServiceWindows: Receive Location NOT created!","Unable to create object for DisableLocation of "+rl.Port+":"+rl.Name + ". Is the condition assembly specified in the configuration file in the GAC?",System.Diagnostics.EventLogEntryType.Error);
							anyErrors = true;
						}
					}
					if (!anyErrors)
						arrLocations.Add(rl);
				}

			}
		}

		/// <summary>
		/// Enable all the ReceiveLocation timers
		/// </summary>
		public void StartProcessing()
		{
			foreach( ReceiveLocation rl in arrLocations)
			{
				rl.Enable();
			}
			EventLog.WriteEntry("ServiceWindowManager Started Processing","");

		}

		/// <summary>
		/// Disable all the ReceiveLocation timers
		/// </summary>
		public void StopProcessing()
		{
			foreach( ReceiveLocation rl in arrLocations)
			{
				rl.Disable();
			}
		}

	}
}
